<?php
/**
 * API: The Rest API class
 *
 * @link        https://oxyprops.com
 *
 * @package     OxyProps
 * @subpackage  API
 * @author      Cédric Bontems <cedric@thewebforge.dev>
 * @since       1.4.0
 * @copyright   Copyright (c) 2022, Cédric Bontems
 * @license     https://www.gnu.org/licenses/gpl-2.0.html  GPLv2 or later
 */

namespace OxyProps\Inc;

use WP_REST_Response;

/**
 * OxyProps\Inc\Rest_Api.
 *
 * Creates endpoints for OxyProps.
 *
 * @since 1.4.0 Creation.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 */
class Rest_Api {


	use Singleton;

	/**
	 * The minimum capability required for reading settings from DB.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $read_settings_minimum_capability Provide a valid WP capability keyword.
	 */
	private string $read_settings_minimum_capability;

	/**
	 * The minimum capability required for saving settings from DB.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $update_settings_minimum_capability Provide a valid WP capability keyword.
	 */
	private string $update_settings_minimum_capability;

	/**
	 * Initialize Rest_Api class.
	 *
	 * Called by the constructor. It retreives and populates all the class properties.
	 *
	 * @see https://developer.wordpress.org/reference/hooks/rest_api_init/
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function register(): void {
		$this->read_settings_minimum_capability   = 'manage_options';
		$this->update_settings_minimum_capability = 'manage_options';
		add_action( 'rest_api_init', array( $this, 'create_rest_routes' ) );
	}

	/**
	 * Creates routes for the oxyprops API.
	 *
	 * This functions registers routes for :
	 * - /settings GET  : Retreive current settings
	 * - /settings POST : Save settings
	 * - /version  GET  : Retreive plugin version
	 * - /license  POST : Activate and deactivate license
	 * - /legacy   GET  : Rettreives legacy information
	 * - /legacy   POST : Clears legacy settings
	 *
	 * @see https://developer.wordpress.org/reference/functions/register_rest_route/
	 * @see https://developer.wordpress.org/rest-api/extending-the-rest-api/adding-custom-endpoints/
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function create_rest_routes() {
		register_rest_route(
			'oxyprops/v1',
			'/settings',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_settings' ),
				'permission_callback' => array( $this, 'get_settings_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/settings',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_settings' ),
				'permission_callback' => array( $this, 'update_settings_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/palette',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_palette' ),
				'permission_callback' => array( $this, 'get_palette_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/palette',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_palette' ),
				'permission_callback' => array( $this, 'update_palette_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/shades',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_shades' ),
				'permission_callback' => array( $this, 'get_shades_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/shades',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_shades' ),
				'permission_callback' => array( $this, 'update_shades_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/version',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_version' ),
				'permission_callback' => array( $this, 'get_version_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/license',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'deactivate' ),
				'permission_callback' => array( $this, 'deactivate_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/legacy',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_legacy' ),
				'permission_callback' => array( $this, 'get_legacy_permission' ),
			)
		);
		register_rest_route(
			'oxyprops/v1',
			'/legacy',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'clear_legacy' ),
				'permission_callback' => array( $this, 'clear_legacy_permission' ),
			)
		);
	}

	/**
	 * Reads the database for the current plugin settings.
	 *
	 * @since 1.4.0 Describe changes.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return WP_REST_Response Current settings.
	 */
	public function get_legacy(): WP_REST_Response {
		$legacy   = get_option( 'oxyprops_canvas_color' );
		$response = $legacy;

		return \rest_ensure_response( $this->success( $response ) );
	}

	/**
	 * Checks if current user is allowed to read settings informations.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return bool Whether the current user is allowed to read settings.
	 */
	public function get_legacy_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Clears the database from legacy settings.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return WP_REST_Response Description.
	 */
	public function clear_legacy(): WP_REST_Response {
		$mapping = array(
			'oxyprops_wants_context'                 => '',
			'oxyprops_wants_elements'                => '',
			'oxyprops_wants_O2_reset'                => '',
			'oxyprops_wants_colors'                  => '',
			'oxyprops_wants_normalize'               => '',
			'oxyprops_wants_user_schemes_stylesheet' => '',
			'oxyprops_wants_colors_utilities'        => '',
			'oxyprops_wants_gradients_utilities'     => '',
			'oxyprops_wants_color_schemes_utilities' => '',
			'oxyprops_wants_shadows_utilities'       => '',
			'oxyprops_wants_aspect_ratios_utilities' => '',
			'oxyprops_wants_typography_utilities'    => '',
			'oxyprops_wants_easing_utilities'        => '',
			'oxyprops_wants_animations_utilities'    => '',
			'oxyprops_wants_sizes_utilities'         => '',
			'oxyprops_wants_borders_utilities'       => '',
			'oxyprops_wants_zindex_utilities'        => '',
			'oxyprops_wants_buttons_utilities'       => '',
			'oxyprops_wants_cards_utilities'         => '',
			'oxyprops_wants_layouts_utilities'       => '',
			'oxyprops_wants_grid12_utilities'        => '',
			'oxyprops_wants_all_utilities'           => '',
			'oxyprops_canvas_color'                  => '',
			'oxyprops_brand_color'                   => '',
			'oxyprops_links_color'                   => '',
			'oxyprops_visited_color'                 => '',
			'OxyProps_lic_email'                     => '',
			'oxyprops_ct_global_settings_bak'        => '',
			'oxyprops_user_custom_canvas'            => '',
			'oxyprops_user_custom_brand'             => '',
			'oxyprops_user_custom_links'             => '',
			'oxyprops_user_custom_visited'           => '',
			'OxyProps_license_Key'                   => '',
			'oxyprops_accent_color'                  => '',
			'oxyprops_user_custom_accent'            => '',
			'oxyprops_font_size_base'                => '',
			'oxyprops_scale_ratio'                   => '',
			'oxyprops_mobile_ratio'                  => '',
			'oxyprops_subheading_ratio'              => '',
			'oxyprops_font_size_stop'                => '',
			'oxyprops_user_custom_canvas_hex'        => '',
			'oxyprops_user_custom_brand_hex'         => '',
			'oxyprops_user_custom_accent_hex'        => '',
			'oxyprops_user_custom_links_hex'         => '',
			'oxyprops_user_custom_visited_hex'       => '',
			'OxyProps_lic_Key'                       => '',
		);
		try {
			foreach ( $mapping as $legacy => $map ) {
				delete_option( $legacy );
			}
			$response = true;
			return \rest_ensure_response( $this->success( $response ) );
		} catch ( \Throwable $th ) {
			$response = false;
			return \rest_ensure_response( $this->error( '500', $response ) );
		}
		$response = false;
		return \rest_ensure_response( $this->error( '400', $response ) );
	}

	/**
	 * Checks if current user is allowed to read settings informations
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return bool Whether the current user has the 'manage_options' capability.
	 */
	public function clear_legacy_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Reads the database for the current plugin settings
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return object WP_REST_Response containing current settings
	 */
	public function get_settings(): WP_REST_Response {
		$current_settings = get_option( 'oxyprops' );
		$response         = $current_settings;

		return \rest_ensure_response( $response );
	}

	/**
	 * Checks if current user is allowed to read palette informations
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.5.0
	 *
	 * @return bool Whether the current user has the 'manage_options' capability.
	 */
	public function get_palette_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Reads the database for the current palette settings
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.5.0
	 *
	 * @return object WP_REST_Response containing current settings
	 */
	public function get_palette(): WP_REST_Response {
		$current_settings = get_option( 'oxyprops' );

		if ( isset( $current_settings['palette'] ) ) {
			$response = $current_settings['palette'];
		} else {
			$defaults = Init::get_instance( 'defaults' );
			$response = $defaults->get_oxyprops_default_settings()['palette'];
		}

		return \rest_ensure_response( $response );
	}

	/**
	 * Checks if current user is allowed to update palette informations
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.5.0
	 *
	 * @return bool Whether the current user has the 'manage_options' capability.
	 */
	public function update_palette_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Updates the palette settings in the database
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.5.0
	 *
	 * @param object $req WP_REST_Request containing the new palette.
	 * @return object WP_REST_Response containing current settings
	 */
	public function update_palette( $req ): WP_REST_Response {
		$current_settings            = get_option( 'oxyprops' );
		$new_palette                 = $req->get_param( 'palette' );
		$current_settings['palette'] = $new_palette;

		$success = update_option( 'oxyprops', $current_settings );

		if ( $success ) {
			$response = 'Successfully saved palette';
		} else {
			$response = 'Error';
		}

		return \rest_ensure_response( $this->success( $response ) );
	}

	/**
	 * Checks if current user is allowed to read shades informations
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.5.0
	 *
	 * @return bool Whether the current user has the 'manage_options' capability.
	 */
	public function get_shades_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Reads the database for the current shades settings
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.5.0
	 *
	 * @return object WP_REST_Response containing current settings
	 */
	public function get_shades(): WP_REST_Response {
		$current_settings = get_option( 'oxyprops' );

		if ( isset( $current_settings['shades'] ) ) {
			$response = $current_settings['shades'];
		} else {
			$defaults = Init::get_instance( 'defaults' );
			$response = $defaults->get_oxyprops_default_settings()['shades'];
		}

		return \rest_ensure_response( $response );
	}

	/**
	 * Checks if current user is allowed to update shades informations
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.5.0
	 *
	 * @return bool Whether the current user has the 'manage_options' capability.
	 */
	public function update_shades_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Updates the shades settings in the database
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.5.0
	 *
	 * @param object $req WP_REST_Request containing the new shades.
	 * @return object WP_REST_Response containing current settings
	 */
	public function update_shades( $req ): WP_REST_Response {
		$current_settings           = get_option( 'oxyprops' );
		$new_shades                 = $req->get_param( 'shades' );
		$current_settings['shades'] = $new_shades;

		$success = update_option( 'oxyprops', $current_settings );

		if ( $success ) {
			$response = 'Successfully saved shades';
		} else {
			$response = 'Error';
		}

		return \rest_ensure_response( $this->success( $response ) );
	}

	/**
	 * Checks if current user is allowed to read settings informations
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return bool Whether the current user has the 'manage_options' capability.
	 */
	public function get_settings_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Gets the plugin version and update info.
	 *
	 * The response contains an array with installed version, latest version
	 * update needed and the license status.
	 *
	 * @author @cbontems
	 * @since 1.4.0 Creation.
	 * @since 1.6.0 Fix Deprecated:  version_compare(): Passing null to parameter #1 ($version1) of type string is deprecated.
	 *
	 * @return WP_REST_Response array('version', 'new_version', 'update', 'license_status').
	 */
	public function get_version(): WP_REST_Response {
		$core                       = Init::get_instance( 'core' );
		$oxyprops                   = Init::get_instance( 'oxyprops' );
		$response                   = array();
		$response['version']        = $oxyprops->get_version() ?? '0.0.0';
		$response['new_version']    = $core->oxyprops_plugin_update_info() ? $core->oxyprops_plugin_update_info()->new_version : $response['version'];
		$response['update']         = version_compare( $response['new_version'], $response['version'], '>' ) ? true : false;
		$response['license_status'] = $core->oxyprops_license_status;
		return \rest_ensure_response( $response );
	}

	/**
	 * Checks if current user is allowed to read settings informations
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return bool Whether the current user has the 'manage_options' capability.
	 */
	public function get_version_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Deactivate license
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @param object $req Request object.
	 *
	 * @return object WP_REST_Response containing current settings
	 */
	public function deactivate( $req ): WP_REST_Response {
		$core     = Init::get_instance( 'core' );
		$response = '';
		$action   = $req->get_param( 'action' );

		switch ( $action ) {
			case 'activate':
				$activated = $core->oxyprops_api_activate_license( $req );
				$activated = $core->oxyprops_check_wordpress_plugin( $req->get_param( 'oxyprops_license_key' ), $req->get_param( 'oxyprops_license_email', $core->oxyprops_license_message, $core->oxyprops_response_object, $core->oxyprops_file ) );
				if ( $activated ) {
					$response = __( 'Your license was successfully activated', 'oxyprops' );
					return \rest_ensure_response( $this->success( $response ) );
				} else {
					$response = __( 'We could not activate your license', 'oxyprops' );
					return \rest_ensure_response( $this->error( '400', $response ) );
				}
				break;
			case 'deactivate':
				$deactivated = $core->oxyprops_api_deactivate_license();
				if ( $deactivated ) {
					$response = __( 'Your license was successfully deactivated', 'oxyprops' );
					return \rest_ensure_response( $this->success( $response ) );
				} else {
					$response = __( 'We could not deactivate your license', 'oxyprops' );
					return \rest_ensure_response( $this->error( '400', $response ) );
				}
				break;
		}
		return \rest_ensure_response( $this->error( '400', __( 'Invalid request', 'oxyprops' ) ) );
	}

	/**
	 * Checks if current user is allowed to deactivvate license
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return bool Whether the current user has the 'manage_options' capability.
	 */
	public function deactivate_permission(): bool {
		return current_user_can( $this->read_settings_minimum_capability );
	}

	/**
	 * Sanitize our options.
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 * @since    1.5.1 removed explicit type declaration for PHP 7.4 compatibility.
	 *
	 * @param string $name The setting name.
	 * @param mixed  $value The value to save.
	 */
	public function sanitize_value( $name, $value ) {
		$callbacks = apply_filters(
			'oxyprops_option_sanitize_callbacks',
			array(
				// Features.
				'context'           => 'rest_sanitize_boolean',
				'elements'          => 'rest_sanitize_boolean',
				'o2_reset'          => 'rest_sanitize_boolean',
				'global_colors'     => 'rest_sanitize_boolean',
				'normalize'         => 'rest_sanitize_boolean',
				'stylesheet'        => 'rest_sanitize_boolean',
				'dequeue_gutenberg' => 'rest_sanitize_boolean',
				'css_print'         => 'sanitize_text_field',

				// Utilities.
				'all'               => 'rest_sanitize_boolean',
				'colors'            => 'rest_sanitize_boolean',
				'gradients'         => 'rest_sanitize_boolean',
				'color_schemes'     => 'rest_sanitize_boolean',
				'shadows'           => 'rest_sanitize_boolean',
				'aspect_ratios'     => 'rest_sanitize_boolean',
				'typography'        => 'rest_sanitize_boolean',
				'easing'            => 'rest_sanitize_boolean',
				'animations'        => 'rest_sanitize_boolean',
				'sizes'             => 'rest_sanitize_boolean',
				'borders'           => 'rest_sanitize_boolean',
				'zindex'            => 'rest_sanitize_boolean',
				'buttons'           => 'rest_sanitize_boolean',
				'cards'             => 'rest_sanitize_boolean',
				'layouts'           => 'rest_sanitize_boolean',
				'grid12'            => 'rest_sanitize_boolean',

				// Typography.
				'base_size'         => 'sanitize_text_field',
				'minimum_size'      => 'sanitize_text_field',
				'scale_ratio'       => 'sanitize_text_field',
				'mobile_ratio'      => 'sanitize_text_field',
				'subheading_ratio'  => 'sanitize_text_field',
			)
		);

		$callback = isset( $callbacks[ $name ] ) ? $callbacks[ $name ] : 'sanitize_text_field';

		if ( ! is_callable( $callback ) ) {
			return sanitize_text_field( $value );
		}

		return $callback( $value );
	}

	/**
	 * Success rest.
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @param mixed $response response data.
	 * @return WP_REST_Response
	 */
	public function success( $response ): WP_REST_Response {
		return new \WP_REST_Response(
			array(
				'success' => true,
				'data'    => $response,
			),
			200
		);
	}

	/**
	 * Error rest.
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @param mixed $code     error code.
	 * @param mixed $response response data.
	 * @return WP_REST_Response
	 */
	public function error( $code, $response ): WP_REST_Response {
		return new \WP_REST_Response(
			array(
				'error'      => true,
				'success'    => false,
				'error_code' => $code,
				'data'       => $response,
			),
			401
		);
	}

	/**
	 * Checks if a multi-levels array contains any value
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @param array $input_array The array to be check for values.
	 * @return bool
	 */
	public function is_array_empty( $input_array ): bool {
		$result = true;

		if ( is_array( $input_array ) && count( $input_array ) > 0 ) {
			foreach ( $input_array as $value ) {
				$result = $result && $this->is_array_empty( $value );
			}
		} else {
			$result = is_array( $input_array ) ? true : ! isset( $input_array );
		}

		return $result;
	}

	/**
	 * Saves the settings to the database through the REST API
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @param object $req request object.
	 * @return WP_REST_Response Success message.
	 */
	public function update_settings( $req ): WP_REST_Response {

		$defaults = Init::get_instance( 'defaults' );
		$oxyprops = Init::get_instance( 'oxyprops' );
		$builder  = Init::get_instance( 'builder' );

		$oxyprops_options          = get_option( 'oxyprops', array() );
		$oxyprops_new_settings     = $req->get_param( 'settings' );
		$default_settings          = $defaults->get_oxyprops_default_settings();
		$oxyprops_current_settings = isset( $oxyprops_options['settings'] ) ? $oxyprops_options['settings'] : array();

		foreach ( $oxyprops_new_settings as $category => $setting ) {

			// Only save categories that we know about.
			if ( ! array_key_exists( $category, $default_settings ) ) {
				unset( $oxyprops_new_settings[ $category ] );
				continue;
			}

			foreach ( $setting as $name => $value ) {

				// First, only save settings that we know about.
				if ( ! array_key_exists( $name, $default_settings[ $category ] ) ) {
					unset( $oxyprops_new_settings[ $category ][ $name ] );
					continue;
				}

				// Then, only keep options that were modified.
				if (
					isset( $oxyprops_current_settings[ $category ][ $name ] ) &&
					$oxyprops_current_settings[ $category ][ $name ] === $oxyprops_new_settings[ $category ][ $name ]
				) {
					unset( $oxyprops_new_settings[ $category ][ $name ] );
					continue;
				}
			}

			if ( count( $oxyprops_new_settings[ $category ] ) < 1 ) {
				unset( $oxyprops_new_settings[ $category ] );
				continue;
			}
		}

		// Finally, Sanitize all remaining leaves in the settings tree.
		$it       = new \RecursiveArrayIterator( $oxyprops_new_settings );
		$iterator = new \RecursiveIteratorIterator( $it );
		foreach ( $iterator as $key => $value ) {
			$value = $this->sanitize_value( $key, $value );
		}

		// Do we need to replace Oxygen default settings ?
		if (
			'Oxygen' === $builder->get_name() &&
			isset( $oxyprops_new_settings['features']['o2_reset'] ) &&
			$oxyprops_new_settings['features']['o2_reset']
		) {
			$helpers = Init::get_instance( 'helpers' );
			$helpers->oxy_settings_override();
		}

		// Do we need to replace Bricks default settings ?
		if (
			'Bricks' === $builder->get_name() &&
			isset( $oxyprops_new_settings['features']['o2_reset'] ) &&
			$oxyprops_new_settings['features']['o2_reset']
		) {
			$helpers = Init::get_instance( 'helpers' );
			$helpers->bricks_settings_override();
		}

		// if no settings were changed, return.
		if ( $this->is_array_empty( $oxyprops_new_settings ) ) {
			return \rest_ensure_response( $this->success( __( 'No changes found in your settings.', 'oxyprops' ) ) );
		}

		// Update settings.
		if ( is_array( $oxyprops_new_settings ) && ! $this->is_array_empty( $oxyprops_new_settings ) ) {
			$updates             = array();
			$updates['settings'] = $oxyprops_new_settings;
			$updated             = update_option( $oxyprops->get_slug(), array_replace_recursive( $oxyprops_options, $updates ) );
			if ( $updated ) {
				return \rest_ensure_response( $this->success( __( 'Settings successfully saved.', 'oxyprops' ) ) );
			}
		}

		return \rest_ensure_response( $this->error( '100', __( 'Something was wrong. Your Settings were not saved.', 'oxyprops' ) ) );
	}

	/**
	 * Checks if current user is allowed to update settings
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return boolean Whether the current user has the 'manage_options' capability.
	 */
	public function update_settings_permission() {
		return current_user_can( $this->update_settings_minimum_capability );
	}
}
